<?php

namespace App\Traits\Gateways;

use App\Models\Deposit;
use App\Models\Withdrawal;
use App\Traits\DepositTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

trait HspayTrait
{
    use DepositTrait;

    /**
     * Initiate a deposit with HSPay.
     * Assumes hosted payment redirect flow until API docs are provided.
     */
    protected function hspay(Deposit $deposit)
    {
        $payload = [
            'item_desc' => setting('site_name'). ' Purchase of '.$deposit->power. ' '. setting('hashpower_unit'). '/s',
            'amount' => (float) number_format($deposit->amount, 8, '.', ''),
            'crypto_1' => setting('deposit_currency_code'),
            'custom_id' => $deposit->invoice,
            'webhook_url' => route('ipn.hspay'),
            'paid_url' => route('ipn.success'),
            'cancel_url' => route('ipn.fail'),
        ];

        // Add optional crypto_2 only if configured
        $crypto2 = setting('hspay_crypto2');
        if (!empty($crypto2)) {
            $payload['crypto_2'] = $crypto2;
        }

        // If a base URL is configured, call HSPay API using api-key header per docs
        $baseUrl = rtrim((string) setting('hspay_base_url', 'https://hspay.io/api/v1'), '/');
        if (!empty($baseUrl)) {
            try {
                $response = Http::withHeaders([
                        'Content-Type' => 'application/json',
                        'api-key' => (string) setting('hspay_merchant_api_key')
                    ])->post($baseUrl . '/merchant/', $payload);
                $body = $response->json();
                // Persist full response for audit and invoice rendering
                $deposit->update(['response' => json_encode($body, JSON_PRETTY_PRINT)]);
                // Accept non-200 only if API indicates success via body format
                if ($response->successful() && isset($body['error']) && $body['error'] === false) {
                    return true;
                }
                // Log and fallback to hosted form if API not available
                logger()->warning('HSPay create payment returned non-success', ['status' => $response->status(), 'body' => $body]);
            } catch (\Throwable $e) {
                logger()->error('HSPay create payment exception: ' . $e->getMessage());
            }
        }

        // Fallback: store minimal context; frontend will post to hosted checkout form
        $deposit->update(['response' => json_encode($payload)]);
        return true;
    }

    /**
     * Handle HSPay IPN/webhook callback.
     * Validates merchant_key, merchant_name, and crypto_1 before processing payment.
     */
    public function hspayIpn(Request $request)
    {
        // Read JSON payload from request body (HSPay sends JSON)
        $payload = $request->json()->all();
        if (!is_array($payload) || empty($payload)) {
            return $this->returnIpnError('Invalid payload format', $request->all());
        }

        $txId = $payload['txid'] ?? null;
        $merchantKey = $payload['merchant_key'] ?? null;
        $merchantName = $payload['merchant_name'] ?? null;
        $itemDesc = $payload['item_desc'] ?? null;
        $amount1 = (float) ($payload['amount_1'] ?? 0);
        $crypto1 = $payload['crypto_1'] ?? null;
        $amount2 = (float) ($payload['amount_2'] ?? 0);
        $crypto2 = $payload['crypto_2'] ?? null;
        $customId = $payload['custom_id'] ?? null;
        $timePay = $payload['time_pay'] ?? null;
        $timeZone = $payload['time_zone'] ?? null;

        // Validate merchant API key
        $myMerchantApi = (string) setting('hspay_merchant_api_key');
        if (empty($myMerchantApi) || $merchantKey !== $myMerchantApi) {
            return $this->returnIpnError('Invalid merchant API key', $payload);
        }

        // Validate merchant name if configured
        $myMerchantName = (string) setting('hspay_merchant_name');
        if (!empty($myMerchantName) && $merchantName !== $myMerchantName) {
            return $this->returnIpnError('Invalid merchant name', $payload);
        }

        // Validate main crypto
        $myMainCrypto = (string) setting('deposit_currency_code');
        if (empty($myMainCrypto) || $crypto1 !== $myMainCrypto) {
            return $this->returnIpnError('Invalid main crypto', 'Expected: ' . $myMainCrypto . ', Got: ' . $crypto1);
        }

        // Find deposit by custom_id (invoice)
        $invoice = $customId;
        $transaction = Deposit::whereInvoice($invoice)->where('status', '!=', 'paid')->first();
        if ($transaction === null) {
            return $this->returnIpnError('Transaction not found. Invoice: #' . $invoice, $payload);
        }

        $this->transaction_id = $transaction->id;
        $this->user_id = $transaction->user_id;

        // Validate amount (use amount_1 as primary check; amount_2 is what they actually paid)
        if ($amount1 < $transaction->amount) {
            return $this->returnIpnError('Amount is less than order total!', 'Amount1: ' . $amount1 . ' - Order Total: ' . $transaction->amount);
        }

        // Update and confirm deposit
        $transaction->update([
            'status' => 'paid',
            'tx_id' => $txId,
            'paid_amount' => $amount2, // Use amount_2 as actual paid amount
            'response' => json_encode($payload, JSON_PRETTY_PRINT),
        ]);

        // Process deposit
        $this->depositConfirmed($transaction);
        return ['status' => 'success', 'message' => 'Payment confirmed'];
    }

    /**
     * Optional: implement withdrawals if HSPay supports it.
     */
    protected function hspayWithdrawal(Withdrawal $withdrawal, $error_reporting)
    {
        try {
            // Validate Faucet API key is configured
            $faucetApiKey = (string) setting('hspay_faucet_api_key');
            if (empty($faucetApiKey)) {
                $errorMsg = 'HSPay Faucet API key is not configured. Please set it in Admin Settings → Gateway → HSPay.';
                logger()->error($errorMsg);
                if ($error_reporting) {
                    return $errorMsg;
                }
                return ['error' => $errorMsg];
            }

            $baseUrl = rtrim((string) setting('hspay_base_url', 'https://hspay.io/api/v1'), '/');
            $endpoint = $baseUrl . '/send-payment/';

            // Choose the user identifier to send to HSPay
            // For withdrawals, default to wallet address as HSPay typically needs it for sending crypto
            $userSource = setting('hspay_user_source', 'wallet'); // email|username|wallet
            $userIdentifier = null;
            
            // Try to get user identifier based on configured source
            if ($userSource === 'wallet' && !empty($withdrawal->user->wallet)) {
                $userIdentifier = $withdrawal->user->wallet;
            } elseif ($userSource === 'username' && !empty($withdrawal->user->username)) {
                $userIdentifier = $withdrawal->user->username;
            } elseif (!empty($withdrawal->user->email)) {
                $userIdentifier = $withdrawal->user->email;
            }
            
            // Fallback: prioritize wallet address for withdrawals (required by most crypto payment gateways)
            if (empty($userIdentifier) && !empty($withdrawal->user->wallet)) {
                $userIdentifier = $withdrawal->user->wallet;
            }
            
            // Validate user identifier is set
            if (empty($userIdentifier)) {
                $errorMsg = 'User identifier (email/wallet) is missing. Please ensure the user has an email or wallet address configured.';
                logger()->error($errorMsg, ['user_id' => $withdrawal->user->id]);
                if ($error_reporting) {
                    return $errorMsg;
                }
                return ['error' => $errorMsg];
            }

            // Validate withdrawal currency is set
            $withdrawalCurrency = setting('withdrawal_currency_code');
            if (empty($withdrawalCurrency)) {
                $errorMsg = 'Withdrawal currency is not configured. Please set it in Admin Settings → General.';
                logger()->error($errorMsg);
                if ($error_reporting) {
                    return $errorMsg;
                }
                return ['error' => $errorMsg];
            }

            // Validate amount is greater than 0
            if (empty($withdrawal->amount) || $withdrawal->amount <= 0) {
                $errorMsg = 'Withdrawal amount must be greater than 0.';
                logger()->error($errorMsg, ['amount' => $withdrawal->amount]);
                if ($error_reporting) {
                    return $errorMsg;
                }
                return ['error' => $errorMsg];
            }

            // Get IP address with fallback
            $ipAddress = request()->server('REMOTE_ADDR') 
                      ?? request()->server('HTTP_X_FORWARDED_FOR') 
                      ?? request()->ip() 
                      ?? '0.0.0.0';

            $payload = [
                'user' => trim($userIdentifier),
                'crypto' => trim($withdrawalCurrency),
                'amount' => (float) number_format($withdrawal->amount, 8, '.', ''),
                'ip_address' => $ipAddress,
            ];
            
            // Validate all payload fields are not empty
            foreach ($payload as $key => $value) {
                if (empty($value) && $key !== 'ip_address') { // ip_address can be 0.0.0.0
                    $errorMsg = "Required field '{$key}' is missing or empty in withdrawal request.";
                    logger()->error($errorMsg, ['payload' => $payload, 'withdrawal_id' => $withdrawal->id]);
                    if ($error_reporting) {
                        return $errorMsg;
                    }
                    return ['error' => $errorMsg];
                }
            }

            logger()->info('HSPay Withdrawal Request', [
                'endpoint' => $endpoint,
                'payload' => $payload,
                'withdrawal_id' => $withdrawal->id,
            ]);

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'api-key' => $faucetApiKey,
            ])->timeout(10)->post($endpoint, $payload);

            $body = $response->json();
            
            // Log full response for debugging
            logger()->info('HSPay Withdrawal Response', [
                'status' => $response->status(),
                'body' => $body,
                'withdrawal_id' => $withdrawal->id,
            ]);

            if ($response->successful() && is_array($body) && isset($body['error']) && $body['error'] === false) {
                $txId = $body['data']['tx_id'] ?? null;
                $paidAmount = $body['data']['amount'] ?? $withdrawal->amount;
                // Update withdrawal
                $this->paidWithdrawal($withdrawal, $body, $txId, $paidAmount);
                // Log
                $this->createWithdrawalUserLog($paidAmount, $withdrawal->user->id);
                return true;
            }

            // Extract error message from response
            $message = 'Unknown error';
            if (is_array($body)) {
                if (isset($body['message'])) {
                    $message = $body['message'];
                } elseif (isset($body['error']) && is_string($body['error'])) {
                    $message = $body['error'];
                } elseif (isset($body['data']['message'])) {
                    $message = $body['data']['message'];
                }
            }
            
            if (empty($message) || $message === 'Unknown error') {
                $message = 'HSPay API returned error. HTTP Status: ' . $response->status();
            }

            payment_log('withdrawal', $message, $withdrawal, $body);
            logger()->error('HSPay Withdrawal Error', [
                'message' => $message,
                'status' => $response->status(),
                'body' => $body,
                'withdrawal_id' => $withdrawal->id,
            ]);

            if ($error_reporting) {
                return $message;
            }
            return ['error' => $message];
        } catch (\Throwable $e) {
            $errorMsg = 'HSPay Exception: ' . $e->getMessage();
            payment_log('withdrawal', $errorMsg, $withdrawal);
            logger()->error($errorMsg, [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'withdrawal_id' => $withdrawal->id,
            ]);
            
            if ($error_reporting) {
                return $errorMsg;
            }
            return ['error' => $errorMsg];
        }
    }
}


