<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Foundation\Http\Middleware\PreventRequestsDuringMaintenance as Middleware;
use Symfony\Component\HttpFoundation\IpUtils;
use Symfony\Component\HttpKernel\Exception\HttpException;

class PreventRequestsDuringMaintenance extends Middleware
{
    /**
     * The URIs that should be reachable while maintenance mode is enabled.
     *
     * @var array
     */
    protected $except = [
        '/assets/*',
        '/_debugbar/*',
    ];

    //Custom Allowed IP's (MOD BY SMARTY SCRIPTS)
    protected $excluded_ips = [];

    public function __construct(Application $app)
    {
        parent::__construct($this->app = $app);

        $urls_allowed = [];
        $ips_allowed = [];

        if(install_status()){
            $urls_allowed = collect(setting('maintenance_excluded_uris'))
                ->mapWithKeys(function ($uri) {
                    return explode(',', $uri);
                })->toArray();

            $ips_allowed = collect(setting('maintenance_excluded_ips'))
                ->mapWithKeys(function ($ip) {
                    return explode(',', $ip);
                })->toArray();
        }

        //Auto add backend to excluded URIs
        $this->except[] = '/'.config('cyber_miner.backend_prefix').'*';
        $this->except = array_merge($this->except, $urls_allowed);
        $this->excluded_ips = $ips_allowed;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     *
     * @throws \Symfony\Component\HttpKernel\Exception\HttpException
     */
    public function handle($request, Closure $next)
    {
        if ($this->app->isDownForMaintenance()) {
            $data = json_decode(file_get_contents($this->app->storagePath() . '/framework/down'), true);

            //Check IP's on down file
            if (isset($data['allowed']) && IpUtils::checkIp($request->ip(), (array)$data['allowed'])) {
                return $next($request);
            }
            //Check IP's on settings
            if (isset($this->excluded_ips) && IpUtils::checkIp($request->ip(), (array)$this->excluded_ips)) {
                return $next($request);
            }

            if (isset($data['secret']) && $request->path() === $data['secret']) {
                return $this->bypassResponse($data['secret']);
            }

            if ($this->hasValidBypassCookie($request, $data) ||
                $this->inExceptArray($request)) {
                return $next($request);
            }

            if (isset($data['redirect'])) {
                $path = $data['redirect'] === '/'
                    ? $data['redirect']
                    : trim($data['redirect'], '/');

                if ($request->path() !== $path) {
                    return redirect($path);
                }
            }

            if (isset($data['template'])) {
                return response(
                    $data['template'],
                    $data['status'] ?? 503,
                    isset($data['retry']) ? ['Retry-After' => $data['retry']] : []
                );
            }

            throw new HttpException(
                $data['status'] ?? 503,
                'Service Unavailable',
                null,
                isset($data['retry']) ? ['Retry-After' => $data['retry']] : []
            );
        }

        return $next($request);
    }
}
